<?php
/* --------------------------------------------------------------
 FilterFactory.php 2020-01-31
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 31 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Admin\Layout\Menu\Factories;

use Gambio\Core\Application\ValueObjects\Path;
use Gambio\Core\Configuration\ConfigurationService;
use Gambio\Admin\Layout\Menu\Filter\FilterCondition;
use Gambio\Admin\Layout\Menu\Filter\FilterConditionArguments;
use Gambio\Admin\Layout\Menu\Filter\FilterFactory as FilterFactoryInterface;
use Gambio\Admin\Layout\Menu\Filter\FilterInterface;
use Gambio\Admin\Layout\Menu\Filter\Types\ConfigActive;
use Gambio\Admin\Layout\Menu\Filter\Types\DisplayOldModuleCenter;
use Gambio\Admin\Layout\Menu\Filter\Types\TemplateVersion;
use InvalidArgumentException;

/**
 * Class FilterFactory
 * @package Gambio\Admin\Layout\Menu\Filter
 */
class FilterFactory implements FilterFactoryInterface
{
    /**
     * @var ConfigurationService
     */
    private $configurationService;
    
    /**
     * @var Path
     */
    private $path;
    
    /**
     * @var bool
     */
    private $debug = false;
    
    
    /**
     * FilterFactory constructor.
     *
     * @param ConfigurationService $configurationService
     * @param Path                 $path
     */
    public function __construct(ConfigurationService $configurationService, Path $path)
    {
        $this->configurationService = $configurationService;
        $this->path                 = $path;
    }
    
    
    /**
     * @inheritDoc
     */
    public function create(FilterCondition $condition): FilterInterface
    {
        if ($condition->filter() === ConfigActive::FILTER_METHOD) {
            return new ConfigActive($this->configurationService);
        }
        
        if ($condition->filter() === TemplateVersion::FILTER_METHOD) {
            return new TemplateVersion($this->configurationService, $this->path);
        }
        
        if ($condition->filter() === DisplayOldModuleCenter::FILTER_METHOD) {
            return new DisplayOldModuleCenter($this->path);
        }
        
        if ($this->debug) {
            throw new InvalidArgumentException("No filter with name ({$condition->filter()}) found");
        }
        
        return $this->failFallback();
    }
    
    
    /**
     * Enables debugging mode.
     */
    public function debug(): void
    {
        $this->debug = true;
    }
    
    
    /**
     * Fallback if no filter was found.
     *
     * This function returns a anonymous class that implements the FilterInterface and
     * return true when using the ::check() method.
     *
     * @return FilterInterface
     */
    private function failFallback(): FilterInterface
    {
        return new class implements FilterInterface {
            public function check(FilterConditionArguments $condition): bool
            {
                return false;
            }
        };
    }
}